package editor;

import game.CustomImageDataII;
import game.IntRect;

import java.awt.Color;
import java.awt.Dimension;
import java.awt.Graphics;
import java.awt.Point;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.util.ArrayList;

import javax.swing.JPanel;
import javax.swing.JScrollPane;

public class ExportablesSelectorPanel extends JPanel {
	
	private ArrayList<ExportablePaletteObject> paletteObjects;
	
	private int paddingBetweenObjects;
	private int paddingFromObjectToBorder;
	private int borderThickness;
	
	private int totalWidth;
	private int totalHeight;
	
	private RoomEditorPanel editorPanel;
	
	private JScrollPane scrollPane;
	
	public ExportablesSelectorPanel(int thePaddingBetweenObjects, int thePaddingToBorder, int theBorderWidth, RoomEditorPanel rep) {
		
		paletteObjects = new ArrayList<ExportablePaletteObject>();
		
		paddingBetweenObjects = thePaddingBetweenObjects;
		paddingFromObjectToBorder = thePaddingToBorder;
		borderThickness = theBorderWidth;
		
		this.setMaximumSize(new Dimension(500, 100));
		this.setMinimumSize(new Dimension(500, 100));
		this.setPreferredSize(new Dimension(500, 100));
		
		totalWidth = paddingBetweenObjects;
		totalHeight = 0;
		
		editorPanel = rep;
		
		this.addMouseListener(new MouseAdapter() {
			public void mouseClicked(MouseEvent e) {
				clickHappened(e);
			}
		});
		
		
	}
	
	
	public void addExportable(PaletteObject po) {
		
		if((po instanceof StaticBlockPaletteObject && !((StaticBlockPaletteObject) po).isNewButton() && !((StaticBlockPaletteObject) po).isEditButton())
				|| po instanceof WidgetScreenPaletteObject
				|| po instanceof RogueBlockScreenPaletteObject
				|| po instanceof HorizontalBlockScreenPaletteObject
				|| po instanceof VerticalBlockScreenPaletteObject
				|| po instanceof ConveyorPaletteObject) {
			//we're good
		} else {
			//skip this one
			return;
		}

		
		
		
		int newbieClickRectMinX = totalWidth + borderThickness + paddingFromObjectToBorder;
		int newbieClickRectMinY = borderThickness + paddingFromObjectToBorder + paddingBetweenObjects;
		
		CustomImageDataII image = po.getImage();
		
		if(image == null) {
//			System.out.println("the image was null!  The type was: " + po.getClass().getCanonicalName());
			if(po instanceof StaticBlockPaletteObject) {
//				System.out.println("block char was: " + ((StaticBlockPaletteObject)po).getBlockChar());
				image = editorPanel.getCharactersToBlocks().get(((StaticBlockPaletteObject)po).getBlockChar()).getImage();
			} else if(po instanceof WidgetScreenPaletteObject) {
				image = ((WidgetScreenPaletteObject)po).getImageArray()[0];
			}
		}
		
		int clickRectDesiredWidth = image.getDesiredWidth();
		int clickRectDesiredHeight = image.getDesiredHeight();
		
		IntRect clickRect = new IntRect(newbieClickRectMinX, newbieClickRectMinY, clickRectDesiredWidth, clickRectDesiredHeight);
		
		ArrayList<String> outputArray = new ArrayList<String>();
		
		if(po instanceof StaticBlockPaletteObject) {
			StaticBlockPaletteObject sbpo = (StaticBlockPaletteObject) po;
			
			if(sbpo.getBlockChar() == ' ' || sbpo.isCloneButton() || sbpo.isNewHurtzoneButton() || sbpo.isNewButton() || sbpo.isEditButton())
				return;
			
			outputArray.add("blockDef:");
			outputArray.add("graphic:" + image.getFileName());
			outputArray.add("transparent:" + CustomAnimationPanel.convertColorToString(image.getTransparentColor()));
			
//			if(editorPanel.getCharactersToBlocks() == null) {
//				System.out.println("null 1");
//			} else if(editorPanel.getCharactersToBlocks().get(sbpo.getBlockChar()) == null) {
//				System.out.println("null 2; char was: " + sbpo.getBlockChar());
//			}
			
			
			if(editorPanel.getCharactersToBlocks().get(sbpo.getBlockChar()).isPassThrough()) {
				outputArray.add("passThru");
			} else {
				outputArray.add("solid");
			}
			outputArray.add("/blockDef");
			
			
		} else if(po instanceof WidgetScreenPaletteObject) {
			
			WidgetScreenPaletteObject wspo = (WidgetScreenPaletteObject) po;
			
			CustomImageDataII imageArray[] = wspo.getImageArray();
			
			outputArray.add("widget:");
			outputArray.add("frameCount:" + imageArray.length);
			for(CustomImageDataII cid : imageArray) {
				outputArray.add(cid.getFileName() + "*" + CustomAnimationPanel.convertColorToString(cid.getTransparentColor()));
			}
			outputArray.add("/widget");
			
		} else if(po instanceof RogueBlockScreenPaletteObject) {
			
			RogueBlockScreenPaletteObject rbspo = (RogueBlockScreenPaletteObject) po;
			
			outputArray.add("block:");
			outputArray.add("width:" + rbspo.getClickRect().width());
			outputArray.add("height:" + rbspo.getClickRect().height());
			outputArray.add("graphic:" + rbspo.getImage().getFileName());
			if(rbspo.shouldBePassThru()) {
				outputArray.add("passThru");
			} else {
				outputArray.add("solid");
			}
			outputArray.add("transparent:" + CustomAnimationPanel.convertColorToString(rbspo.getImage().getTransparentColor()));
			outputArray.add("/block");
			
		} else if(po instanceof HorizontalBlockScreenPaletteObject) {
			
			HorizontalBlockScreenPaletteObject hbspo = (HorizontalBlockScreenPaletteObject) po;
			
			outputArray.add("movingBlock:");
			outputArray.add("width:" + hbspo.getClickRect().width());
			outputArray.add("height:" + hbspo.getClickRect().height());
			outputArray.add("graphic:" + hbspo.getImage().getFileName());
			if(hbspo.shouldBePassThru()) {
				outputArray.add("passThru");
			} else {
				outputArray.add("solid");
			}
			outputArray.add("transparent:" + CustomAnimationPanel.convertColorToString(hbspo.getImage().getTransparentColor()));
			outputArray.add("horizontal");
			outputArray.add("defaultSpeed:" + hbspo.getDefaultSpeed());
			outputArray.add("/movingBlock");
			
		} else if(po instanceof VerticalBlockScreenPaletteObject) {
			
			VerticalBlockScreenPaletteObject hbspo = (VerticalBlockScreenPaletteObject) po;
			
			outputArray.add("movingBlock:");
			outputArray.add("width:" + hbspo.getClickRect().width());
			outputArray.add("height:" + hbspo.getClickRect().height());
			outputArray.add("graphic:" + hbspo.getImage().getFileName());
			if(hbspo.shouldBePassThru()) {
				outputArray.add("passThru");
			} else {
				outputArray.add("solid");
			}
			outputArray.add("transparent:" + CustomAnimationPanel.convertColorToString(hbspo.getImage().getTransparentColor()));
			outputArray.add("vertical");
			outputArray.add("defaultSpeed:" + hbspo.getDefaultSpeed());
			outputArray.add("/movingBlock");
			
		} else if(po instanceof ConveyorPaletteObject) {
			
			ConveyorPaletteObject cpo = (ConveyorPaletteObject) po;
			
			outputArray.add("conveyor:");
			outputArray.add("width:" + cpo.getClickRect().width());
			outputArray.add("height:" + cpo.getClickRect().height());
			if(cpo.isPassThru()) {
				outputArray.add("passThru");
			} else {
				outputArray.add("solid");
			}
			if(cpo.isGoingLeft()) {
				outputArray.add("dir:l");
			} else {
				outputArray.add("dir:r");
			}
			outputArray.add("speed:" + cpo.getSpeed());
			
			CustomImageDataII imageArray[] = cpo.getAnimation();
			
			outputArray.add("frameCount:" + imageArray.length);
			for(CustomImageDataII cid : imageArray) {
				outputArray.add(cid.getFileName() + "*" + CustomAnimationPanel.convertColorToString(cid.getTransparentColor()));
			}
			outputArray.add("/block");
			
		}
		
		
		ExportablePaletteObject epo = new ExportablePaletteObject(clickRect, image, paddingFromObjectToBorder, borderThickness, editorPanel, outputArray.toArray(new String[0]));
		
		paletteObjects.add(epo);
		
		int maxHeight = this.getHeightOfLargestItem();
		
		totalWidth += epo.getClickRect().width() + borderThickness * 2 + paddingBetweenObjects + paddingFromObjectToBorder * 2;
		if(totalWidth > this.getPreferredSize().width || totalHeight < maxHeight) {
			
			totalHeight = maxHeight + borderThickness * 2 + paddingBetweenObjects * 2 + paddingFromObjectToBorder * 2;
			
			int newWidth = Math.max(500, totalWidth);
			int newHeight = Math.max(100, totalHeight);
			
			this.setPreferredSize(new Dimension(newWidth, newHeight));
			this.setMinimumSize(new Dimension(newWidth, newHeight));
			this.setMaximumSize(new Dimension(newWidth, newHeight));
			
		}
		
		if(scrollPane != null) {
//			scrollPane.repaint();
			scrollPane.revalidate();
		}
		
	}
	
	
	private int getHeightOfLargestItem() {
		int returnValue = Integer.MIN_VALUE;
		
		for(ExportablePaletteObject epo : paletteObjects ) {
			if(returnValue < epo.getClickRect().height())
				returnValue = epo.getClickRect().height();
		}
		
		return returnValue;
	}


	public JScrollPane getScrollPane() {
		return scrollPane;
	}


	public void setScrollPane(JScrollPane scrollPane) {
		this.scrollPane = scrollPane;
	}
	
	
	public void paintComponent(Graphics g) {
		
		Color origCol = g.getColor();
		
		g.setColor(Color.white);
		g.fillRect(0, 0, this.getSize().width, this.getSize().height);
		g.setColor(origCol);
		
		for(ExportablePaletteObject epo : this.paletteObjects) {
			epo.draw(g);
		}
		
		g.setColor(origCol);
		
	}
	
	public void clickHappened(MouseEvent e) {
		
		Point clickPoint = new Point(e.getX(), e.getY());
		
		for(ExportablePaletteObject epo : this.paletteObjects) {
			
			if(epo.getClickRect() != null && epo.getClickRect().contains(clickPoint)) {
				epo.setSelected(!epo.isSelected());
				break;
			}
			
		}
		
		this.repaint();
		
	}
	
	public ArrayList<ExportablePaletteObject> getList() {
		return paletteObjects;
	}
	
}
